import { Tool } from "@modelcontextprotocol/sdk/types.js";

/**
 * MCP tool definitions for Ssalmuk character management.
 * 공식 배포용으로 설명을 중립적이고 정책 친화적으로 정리했습니다.
 */

const characterCreateDescription =
  "쌀먹챗 캐릭터 초안(Character Card)을 생성하는 도구입니다.\n" +
  "name, description, startingSets(최소 1개)를 필수로 받고,\n" +
  "customPrompt, 이미지 설정, 예시 대화, 해시태그 등은 선택적으로 제공할 수 있습니다.\n\n" +
  "이미지 관련 필드는 Danbooru/NAI 스타일 태그 등 다양한 형식을 사용할 수 있지만,\n" +
  "항상 각 서비스와 플랫폼의 이용 약관 및 콘텐츠·안전 가이드를 준수해야 합니다.\n\n" +
  "또한 쌀먹챗 앱의 '명령어' 기능과 잘 연동될 수 있도록, 느낌표(!)로 시작하는 채팅 명령어 후보를 함께 설계해 두면 좋습니다.\n" +
  "예: !요약(최근 대화 요약), !디버그(오류 신고), !공략(다음 행동 가이드), !사칭방지(사칭/피싱 방지 안내) 등은\n" +
  "캐릭터 성격과 세계관에 맞게 정의하여 앱의 명령어 탭에서 등록해 사용할 수 있습니다.";

export const characterCreateTool: Tool = {
  name: "character_create",
  description: characterCreateDescription,
  inputSchema: {
    type: "object",
    properties: {
      name: {
        type: "string",
        description: "캐릭터 제목 (앱에 표시되는 이름, 최대 20자)",
      },
      description: {
        type: "string",
        description: "캐릭터를 한 줄~두 줄로 설명하는 문장 (최대 1000자)",
      },
      customPrompt: {
        type: "string",
        description:
          "캐릭터의 시스템 프롬프트/룰북 역할을 하는 상세 설명입니다. 선택 항목이며 5500자 이내를 권장합니다.",
      },
      target: {
        type: "string",
        enum: ["all", "male", "female"],
        description: "주 타깃 사용자",
        default: "all",
      },
      isAdult: {
        type: "boolean",
        description:
          '성인 작품 여부입니다. true면 audienceTier가 "adultsOnly"로 설정됩니다.',
        default: false,
      },
      profileImageUrl: {
        type: "string",
        description:
          "대표 이미지 URL (선택). 비워 두면 앱에서 기본 위젯으로 표시됩니다.",
      },
      advancedImageSettings: {
        type: "object",
        description: "고급 이미지 생성 옵션 (선택).",
        properties: {
          enabled: {
            type: "boolean",
            description: "이미지 생성 사용 여부",
            default: true,
          },
          model: {
            type: "string",
            description: "이미지 생성 모델 (예: nai-diffusion-4-5-full)",
            default: "nai-diffusion-4-5-full",
          },
          sampler: {
            type: "string",
            description: "샘플러 (예: k_euler_ancestral)",
            default: "k_euler_ancestral",
          },
          cfgScale: {
            type: "number",
            description: "CFG Scale",
            default: 6.5,
          },
          steps: {
            type: "number",
            description: "생성 스텝 수",
            default: 28,
          },
          alwaysAddedPrompt: {
            type: "string",
            description:
              "모든 이미지 프롬프트에 공통으로 추가할 품질/스타일 태그입니다. 선택 항목입니다.",
          },
          negativePrompt: {
            type: "string",
            description: "공통 네거티브 프롬프트입니다. 선택 항목입니다.",
          },
          profilePrompt: {
            type: "string",
            description:
              "대표 이미지를 위한 태그 나열입니다. 영어 Danbooru/NAI 스타일을 권장합니다.",
          },
          profileAspect: {
            type: "string",
            enum: ["square", "portrait", "landscape"],
            description:
              '대표 이미지 비율 (허용: "square", "portrait", "landscape" - 예: 1:1, 4:3 같은 수치는 불가)',
            default: "square",
          },
        },
      },
      specialType: {
        type: "string",
        enum: ["card_builder"],
        description: "특수 타입 값입니다. 일반적으로 생략합니다.",
      },
      hashtags: {
        type: "array",
        items: { type: "string" },
        description:
          "해시태그 목록 (# 없이, 2~20자, 한글/영어·띄어쓰기 허용). 0~10개 권장.",
        maxItems: 10,
      },
      genreName: {
        type: "string",
        enum: [
          "로맨스",
          "로판",
          "SF/판타지",
          "일상/현대",
          "무협",
          "시대",
          "BL",
          "GL",
          "기타",
        ],
        description: "장르 (선택)입니다.",
      },
      keywordNotes: {
        type: "array",
        description:
          "키워드 기반 로어북 항목입니다. 특정 단어가 등장했을 때 추가로 붙일 설정입니다.",
        maxItems: 10,
        items: {
          type: "object",
          properties: {
            name: {
              type: "string",
              description: "로어북 항목 이름입니다.",
            },
            prompt: {
              type: "string",
              description: "해당 키워드가 트리거될 때 추가할 설명입니다 (5~400자 권장).",
            },
            keywords: {
              type: "array",
              items: { type: "string" },
              description: "트리거 키워드 목록입니다.",
              maxItems: 5,
            },
            targets: {
              type: "array",
              items: { type: "number" },
              description: "적용할 startingSets 인덱스 배열입니다 (선택).",
            },
          },
          required: ["name", "prompt", "keywords"],
        },
      },
      commandPresets: {
        type: "array",
        description:
          "채팅 입력창에서 사용할 명령어 목록입니다. 각 명령어는 보통 '!'로 시작하는 트리거와 상세 동작 설명으로 구성됩니다.",
        maxItems: 20,
        items: {
          type: "object",
          properties: {
            id: {
              type: "string",
              description:
                "명령어 고유 ID (선택). 제공하지 않으면 서버가 자동으로 생성합니다.",
            },
            command: {
              type: "string",
              description:
                "채팅창에 입력할 실제 명령어 트리거입니다. 보통 느낌표(!)로 시작하는 한 단어를 권장합니다. 예: '!요약', '!디버그'.",
            },
            description: {
              type: "string",
              description:
                "채팅 입력창에 함께 표시할 안내 문구입니다. 사용자가 명령어 용도를 이해할 수 있도록 간단히 적어 주세요 (최대 80자 권장).",
            },
            instruction: {
              type: "string",
              description:
                "명령어가 호출되면 시스템 프롬프트에 추가될 상세 지시입니다. 이 지시에 따라 캐릭터가 어떻게 행동할지 정의합니다 (5자 이상 권장).",
            },
          },
          required: ["command", "instruction"],
        },
      },
      startingSets: {
        type: "array",
        description:
          "첫 메세지 목록입니다. 최소 1개 이상의 첫 메세지가 필요합니다.",
        minItems: 1,
        items: {
          type: "object",
          properties: {
            name: {
              type: "string",
              description: "첫 메세지 이름입니다.",
            },
            situationPrompt: {
              type: "string",
              description: "시작 상황 설명입니다 (선택).",
            },
            playGuide: {
              type: "string",
              description:
                "플레이 가이드 안내입니다. 해당 캐릭터를 플레이할 유저들을 위한 힌트나 가이드입니다. (선택).",
            },
            initialMessages: {
              type: "array",
              description:
                "프롤로그 메시지 배열입니다. 일반적으로 assistant 역할 메시지 1개 이상을 권장합니다(각 메시지 2~2000자 권장).",
              minItems: 1,
              items: {
                type: "object",
                properties: {
                  role: {
                    type: "string",
                    enum: ["assistant", "user", "system"],
                    description: "메시지 화자입니다.",
                  },
                  text: {
                    type: "string",
                    description: "메시지 내용입니다.",
                  },
                },
                required: ["role", "text"],
              },
            },
            replySuggestions: {
              type: "array",
              items: { type: "string" },
              description:
                "사용자에게 보여줄 추천 입력 문장입니다 (선택, 최대 5개).",
              maxItems: 5,
            },
            situationImages: {
              type: "array",
              description:
                "상황별 이미지 플레이스홀더입니다. 각 항목은 키워드, 설명, 비율, 프롬프트를 포함합니다.",
              items: {
                type: "object",
                properties: {
                  imageUrl: {
                    type: "string",
                    description: "샘플 이미지 URL입니다 (선택).",
                  },
                  keyword: {
                    type: "string",
                    description: "이미지를 대표하는 키워드입니다 (최대 30자).",
                  },
                  situation: {
                    type: "string",
                    description: "짧은 상황 설명입니다 (최대 50자).",
                  },
                  aspect: {
                    type: "string",
                    enum: ["square", "portrait", "landscape"],
                    description:
                      '이미지 비율 (허용: "square", "portrait", "landscape" - 예: 1:1, 4:3 같은 수치는 불가)',
                  },
                  prompt: {
                    type: "string",
                    description:
                      "해당 상황을 잘 나타내는 Danbooru/NAI 스타일 태그 기반 프롬프트입니다.",
                  },
                },
                required: ["keyword", "situation", "aspect", "prompt"],
              },
            },
          },
          required: ["name", "initialMessages"],
        },
      },
    },
    required: ["name", "description", "startingSets"],
  },
};

export const characterListTool: Tool = {
  name: "character_list",
  description: "미등록 상태(draft)의 캐릭터 목록을 조회합니다.",
  inputSchema: {
    type: "object",
    properties: {
      limit: {
        type: "number",
        description: "가져올 최대 개수 (기본값: 20).",
        default: 20,
      },
      offset: {
        type: "number",
        description: "건너뛸 개수 (페이징용, 기본값: 0).",
        default: 0,
      },
    },
  },
};

export const publicCharacterListTool: Tool = {
  name: "public_character_list",
  description: "현재 계정의 공개 캐릭터를 최신순으로 조회합니다.",
  inputSchema: {
    type: "object",
    properties: {
      limit: {
        type: "number",
        description: "가져올 최대 개수 (기본값: 20).",
        default: 20,
      },
      offset: {
        type: "number",
        description: "건너뛸 개수 (페이징용, 기본값: 0).",
        default: 0,
      },
      detail: {
        type: "boolean",
        description:
          "상세 필드(startingSets, commandPresets, 대표 댓글)를 함께 가져올지 여부입니다. 기본값: false.",
        default: false,
      },
    },
  },
};

export const characterGetTool: Tool = {
  name: "character_get",
  description: "특정 캐릭터 draft의 상세 정보를 조회합니다.",
  inputSchema: {
    type: "object",
    properties: {
      seriesId: {
        type: "string",
        description: "캐릭터 draft series ID입니다.",
      },
    },
    required: ["seriesId"],
  },
};

export const allTools: Tool[] = [
  characterCreateTool,
  characterListTool,
  publicCharacterListTool,
  characterGetTool,
];
