import { Server } from "@modelcontextprotocol/sdk/server/index.js";
import { StdioServerTransport } from "@modelcontextprotocol/sdk/server/stdio.js";
import {
  CallToolRequestSchema,
  ListToolsRequestSchema,
} from "@modelcontextprotocol/sdk/types.js";
import { allTools } from "./tools.js";
import { ensureApiKeyVerified, isAuthenticated } from "./auth.js";
import {
  createCharacterDraft,
  listCharacterDrafts,
  getCharacterDraft,
  listPublicCharacters,
} from "./database.js";
import {
  normalizeCharacterCard,
  generateDefaultKeywordNotes,
} from "./validation.js";
import type { CharacterCard } from "./types.js";
import { McpApiError } from "./httpClient.js";
import { VERSION } from "./version.js";

/**
 * MCP Server for Ssalmuk Character Management
 */
export class SsalmukCharacterServer {
  private server: Server;

  constructor() {
    this.server = new Server(
      {
        name: "ssalmuk-character-mcp",
        version: VERSION,
      },
      {
        capabilities: {
          tools: {},
        },
      }
    );

    this.setupHandlers();
  }

  private setupHandlers() {
    // List available tools
    this.server.setRequestHandler(ListToolsRequestSchema, async () => ({
      tools: allTools,
    }));

    // Handle tool calls
    this.server.setRequestHandler(CallToolRequestSchema, async (request) => {
      const { name, arguments: args } = request.params;

      try {
        switch (name) {
          case "character_create":
            return await this.handleCharacterCreate(args);
          case "character_list":
            return await this.handleCharacterList(args);
          case "public_character_list":
            return await this.handlePublicCharacterList(args);
          case "character_get":
            return await this.handleCharacterGet(args);
          default:
            throw new Error(`Unknown tool: ${name}`);
        }
      } catch (error) {
        const errorMessage = formatToolError(error);
        return {
          content: [
            {
              type: "text",
              text: `Error: ${errorMessage}`,
            },
          ],
        };
      }
    });
  }

  private async handleCharacterCreate(rawArgs: unknown) {
    if (!isAuthenticated()) {
      throw new Error(
        "API 키가 인증되지 않았습니다. .env의 MCP_API_KEY를 확인해 주세요."
      );
    }

    const args = ensureObject(rawArgs, "character_create");

    // Strict: 루트 level situationImages는 허용하지 않는다.
    if (Array.isArray((args as any).situationImages)) {
      throw new Error(
        "situationImages는 각 startingSets[i].situationImages 안에 넣어야 합니다. 루트에는 필드가 없습니다."
      );
    }

    // Add default keyword notes if not provided
    if (!args.keywordNotes || args.keywordNotes.length === 0) {
      args.keywordNotes = generateDefaultKeywordNotes();
    }

    // Normalize and validate the character card
    const character = normalizeCharacterCard(args as Partial<CharacterCard>);

    // Save to database
    const result = await createCharacterDraft(character);

    return {
      content: [
        {
          type: "text",
          text: JSON.stringify(
            {
              success: true,
              ...result,
              character: {
                name: character.name,
                description: character.description,
                target: character.target,
              },
            },
            null,
            2
          ),
        },
      ],
    };
  }

  private async handleCharacterList(rawArgs: unknown) {
    if (!isAuthenticated()) {
      throw new Error(
        "API 키가 인증되지 않았습니다. .env의 MCP_API_KEY를 확인해 주세요."
      );
    }

    const { limit, offset } = parsePaginationArgs(rawArgs);

    const drafts = await listCharacterDrafts({
      limit,
      offset,
    });

    return {
      content: [
        {
          type: "text",
          text: JSON.stringify(
            {
              success: true,
              count: drafts.length,
              drafts: drafts.map((d) => ({
                id: d.id,
                title: d.title,
                description: d.description,
                created_at: d.created_at,
                updated_at: d.updated_at,
              })),
            },
            null,
            2
          ),
        },
      ],
    };
  }

  private async handlePublicCharacterList(rawArgs: unknown) {
    if (!isAuthenticated()) {
      throw new Error(
        "API 키가 인증되지 않았습니다. .env의 MCP_API_KEY를 확인해 주세요."
      );
    }

    const { limit, offset, detail } = parsePublicCharacterListArgs(rawArgs);
    const characters = await listPublicCharacters({ limit, offset, detail });

    return {
      content: [
        {
          type: "text",
          text: JSON.stringify(
            {
              success: true,
              count: characters.length,
              characters: characters.map((c) => ({
                id: c.id,
                name: c.name,
                description: c.description,
                visibility: c.visibility,
                is_adult: c.is_adult,
                tags: c.tags,
                created_at: c.created_at,
                updated_at: c.updated_at,
                ...(detail
                  ? {
                      profileImage: c.profileImage,
                      startingSets: c.startingSets ?? [],
                      commandPresets: c.commandPresets ?? [],
                      representativeComment: c.representativeComment ?? null,
                    }
                  : {}),
              })),
            },
            null,
            2
          ),
        },
      ],
    };
  }

  private async handleCharacterGet(rawArgs: unknown) {
    if (!isAuthenticated()) {
      throw new Error(
        "API 키가 인증되지 않았습니다. .env의 MCP_API_KEY를 확인해 주세요."
      );
    }

    const { seriesId } = parseCharacterGetArgs(rawArgs);

    const { series, version } = await getCharacterDraft(seriesId);

    // Convert character to assistant in chatExamples for LLM compatibility
    // Use loose typing to keep compatibility with older drafts that may include chatExamples.
    const characterPayload: any = { ...version.payload };
    if (characterPayload.chatExamples && Array.isArray(characterPayload.chatExamples)) {
      characterPayload.chatExamples = characterPayload.chatExamples.map((example: any) => {
        if (example.character !== undefined) {
          return {
            user: example.user,
            assistant: example.character, // Convert character to assistant for LLM
          };
        }
        return example;
      });
    }

    return {
      content: [
        {
          type: "text",
          text: JSON.stringify(
            {
              success: true,
              series: {
                id: series.id,
                title: series.title,
                description: series.description,
                status: series.status,
                created_at: series.created_at,
                updated_at: series.updated_at,
              },
              version: {
                id: version.id,
                version_number: version.version_number,
                created_at: version.created_at,
              },
              character: characterPayload,
            },
            null,
            2
          ),
        },
      ],
    };
  }

  async run() {
    await ensureApiKeyVerified();
    const transport = new StdioServerTransport();
    await this.server.connect(transport);
    console.error("Ssalmuk Character MCP Server running on stdio");
  }
}

function formatToolError(error: unknown): string {
  if (error instanceof McpApiError) {
    const status =
      error.status > 0 ? `(HTTP ${error.status})` : "(Network error)";
    const requestId = error.requestId ? ` requestId=${error.requestId}` : "";
    return `${error.message} ${status}${requestId}`.trim();
  }

  if (error instanceof Error) {
    return error.message;
  }

  return String(error);
}

function ensureObject(value: unknown, toolName: string): Record<string, any> {
  if (value && typeof value === "object") {
    return value as Record<string, any>;
  }
  throw new Error(`Invalid arguments supplied to ${toolName}`);
}

function parsePaginationArgs(
  rawArgs: unknown
): { limit: number; offset: number } {
  if (!rawArgs || typeof rawArgs !== "object") {
    return { limit: 20, offset: 0 };
  }

  const input = rawArgs as Record<string, unknown>;
  const limit = clampInteger(input.limit, 1, 50) ?? 20;
  const offset = clampInteger(input.offset, 0, 1000) ?? 0;
  return { limit, offset };
}

function parsePublicCharacterListArgs(
  rawArgs: unknown
): { limit: number; offset: number; detail: boolean } {
  const { limit, offset } = parsePaginationArgs(rawArgs);
  let detail = false;

  if (rawArgs && typeof rawArgs === "object") {
    const value = (rawArgs as any).detail;
    if (typeof value === "string") {
      const normalized = value.trim().toLowerCase();
      detail = ["1", "true", "yes", "y", "on"].includes(normalized);
    } else {
      detail = Boolean(value);
    }
  }

  return { limit, offset, detail };
}

function parseCharacterGetArgs(rawArgs: unknown): { seriesId: string } {
  if (!rawArgs || typeof rawArgs !== "object") {
    throw new Error("seriesId is required");
  }

  const seriesId = typeof (rawArgs as any).seriesId === "string"
    ? (rawArgs as any).seriesId.trim()
    : "";

  if (!seriesId) {
    throw new Error("seriesId is required");
  }

  return { seriesId };
}

function clampInteger(
  value: unknown,
  min: number,
  max: number
): number | undefined {
  const parsed = parseInteger(value);
  if (parsed === undefined) {
    return undefined;
  }
  return Math.min(Math.max(parsed, min), max);
}

function parseInteger(value: unknown): number | undefined {
  if (typeof value === "number" && Number.isFinite(value)) {
    return Math.trunc(value);
  }
  if (typeof value === "string" && value.trim().length > 0) {
    const parsed = Number.parseInt(value, 10);
    if (!Number.isNaN(parsed)) {
      return parsed;
    }
  }
  return undefined;
}

function parseEnum<T extends string>(
  value: unknown,
  allowed: readonly T[]
): T | undefined {
  if (typeof value !== "string") {
    return undefined;
  }
  const normalized = value.trim();
  return allowed.includes(normalized as T) ? (normalized as T) : undefined;
}
