import { callMcpApi } from "./httpClient.js";
import type {
  CharacterCard,
  CharacterDraftSeries,
  CharacterDraftVersion,
  PublicCharacterSummary,
} from "./types.js";

/**
 * Create a new character draft (unregistered state)
 * Uses Edge Function to bypass RLS
 */
export async function createCharacterDraft(
  character: CharacterCard
): Promise<{ seriesId: string; versionId: string; message: string }> {
  const data = await callMcpApi<{
    success: boolean;
    seriesId: string;
    versionId: string;
    message: string;
    error?: string;
  }>("characters", {
    method: "POST",
    body: { character },
  });

  if (!data?.success) {
    throw new Error(data?.error || "Failed to create character");
  }

  return {
    seriesId: data.seriesId,
    versionId: data.versionId,
    message: data.message,
  };
}

/**
 * List user's character drafts
 * Uses Edge Function to bypass RLS
 */
export async function listCharacterDrafts(options?: {
  limit?: number;
  offset?: number;
}): Promise<CharacterDraftSeries[]> {
  const limit = options?.limit || 20;
  const offset = options?.offset || 0;

  const data = await callMcpApi<{
    success: boolean;
    drafts: CharacterDraftSeries[];
    error?: string;
  }>("characters", {
    method: "GET",
    query: {
      limit,
      offset,
    },
  });

  if (!data?.success) {
    throw new Error(data?.error || "Failed to list characters");
  }

  return data.drafts || [];
}

/**
 * List public characters (published)
 */
export async function listPublicCharacters(options?: {
  limit?: number;
  offset?: number;
  detail?: boolean;
}): Promise<PublicCharacterSummary[]> {
  const limit = options?.limit || 20;
  const offset = options?.offset || 0;
  const detail = Boolean(options?.detail);

  const query: Record<string, any> = {
    limit,
    offset,
  };

  if (detail) {
    query.detail = "true";
  }

  const data = await callMcpApi<{
    success: boolean;
    characters: PublicCharacterSummary[];
    error?: string;
  }>("public-characters", {
    method: "GET",
    query,
  });

  if (!data?.success) {
    throw new Error(data?.error || "Failed to list public characters");
  }

  return data.characters || [];
}

/**
 * Get a specific character draft with its latest version
 * Uses Edge Function to bypass RLS
 */
export async function getCharacterDraft(seriesId: string): Promise<{
  series: CharacterDraftSeries;
  version: CharacterDraftVersion;
}> {
  const data = await callMcpApi<{
    success: boolean;
    series: CharacterDraftSeries;
    version: CharacterDraftVersion;
    error?: string;
  }>(`characters/${seriesId}`, {
    method: "GET",
  });

  if (!data?.success) {
    throw new Error(data?.error || "Failed to get character");
  }

  return { series: data.series, version: data.version };
}
