const DEFAULT_ADVANCED_IMAGE_SETTINGS = {
    enabled: true,
    model: "nai-diffusion-4-5-full",
    sampler: "k_euler_ancestral",
    cfgScale: 6.5,
    steps: 28,
    profilePrompt: "",
    profileAspect: "square",
    situationPrompts: [],
};
function normalizeAudienceTier(audienceTier, isAdult) {
    if (audienceTier === "adultsOnly" || audienceTier === "allAges") {
        return audienceTier;
    }
    if (isAdult === true) {
        return "adultsOnly";
    }
    return "allAges";
}
/**
 * Validates and normalizes a character card with default values
 */
export function normalizeCharacterCard(input) {
    // Ensure required fields
    if (!input.name || input.name.trim().length === 0) {
        throw new Error("Character name is required");
    }
    if (!input.description || input.description.trim().length === 0) {
        throw new Error("Character description is required");
    }
    if (!input.startingSets || input.startingSets.length === 0) {
        throw new Error("At least one starting set is required");
    }
    // Normalize starting sets
    const normalizedStartingSets = input.startingSets.map((set, index) => normalizeStartingSet(set, index));
    const audienceTier = normalizeAudienceTier(input.audienceTier, input.isAdult);
    // Build normalized character card
    const normalized = {
        name: input.name.trim(),
        description: input.description.trim(),
        customPrompt: input.customPrompt?.trim() || "",
        target: input.target || "all",
        audienceTier,
        isAdult: audienceTier === "adultsOnly",
        profileImageUrl: input.profileImageUrl?.trim() || "",
        advancedImageSettings: normalizeAdvancedImageSettings(input.advancedImageSettings),
        specialType: input.specialType || undefined,
        hashtags: input.hashtags || [],
        genreName: input.genreName || undefined,
        keywordNotes: input.keywordNotes || [],
        commandPresets: normalizeCommandPresets(input.commandPresets),
        startingSets: normalizedStartingSets,
    };
    const enriched = syncSituationPrompts(normalized);
    if (!enriched.advancedImageSettings?.situationPrompts?.length) {
        throw new Error("At least one situation image prompt is required. Provide prompt text for each image.");
    }
    return enriched;
}
function normalizeCommandPresets(presets) {
    if (!presets || presets.length === 0) {
        return [];
    }
    const normalized = [];
    presets.forEach((raw, index) => {
        const command = typeof raw.command === "string" ? raw.command.trim() : "";
        const instruction = typeof raw.instruction === "string" ? raw.instruction.trim() : "";
        if (!command || !instruction) {
            // Skip incomplete entries rather than failing the whole request.
            return;
        }
        // In the app UX 명령어는 보통 느낌표로 시작하므로,
        // MCP에서도 기본적으로 없으면 붙여 준다.
        const normalizedCommand = command.startsWith("!")
            ? command
            : `!${command}`;
        const normalizedName = normalizedCommand;
        normalized.push({
            id: typeof raw.id === "string" && raw.id.trim().length > 0
                ? raw.id.trim()
                : `cmd_${index + 1}`,
            name: normalizedName,
            command: normalizedCommand,
            description: typeof raw.description === "string"
                ? raw.description.trim()
                : undefined,
            instruction,
        });
    });
    return normalized;
}
function normalizeStartingSet(set, index) {
    if (!set.name || set.name.trim().length === 0) {
        throw new Error(`Starting set #${index + 1} must have a name`);
    }
    if (!set.initialMessages || set.initialMessages.length === 0) {
        throw new Error(`Starting set "${set.name}" must have at least one initial message`);
    }
    if (!set.situationImages || set.situationImages.length === 0) {
        throw new Error(`Starting set "${set.name}" must include at least one situation image (startingSets[${index}].situationImages)`);
    }
    const normalizedMessages = set.initialMessages.map((msg, msgIndex) => normalizeInitialMessage(msg, msgIndex));
    const normalizedImages = set.situationImages.map((img, imgIndex) => normalizeSituationImage(img, index, imgIndex));
    const playGuide = typeof set.playGuide === "string" ? set.playGuide.trim() : "";
    const clampedPlayGuide = playGuide.length > 400 ? playGuide.slice(0, 400) : playGuide;
    return {
        id: set.id || (index === 0 ? "default" : `setting_${index}`),
        name: set.name.trim(),
        situationPrompt: set.situationPrompt?.trim() || "",
        playGuide: clampedPlayGuide,
        initialMessages: normalizedMessages,
        replySuggestions: set.replySuggestions || [],
        situationImages: normalizedImages,
    };
}
function normalizeInitialMessage(msg, index) {
    if (!msg.text || msg.text.trim().length === 0) {
        throw new Error(`Initial message #${index + 1} cannot be empty`);
    }
    const role = msg.role || "assistant";
    if (!["assistant", "user", "system"].includes(role)) {
        throw new Error(`Invalid role "${role}" in initial message`);
    }
    return {
        id: msg.id || `msg_${Date.now()}_${index}`,
        role: role,
        text: msg.text.trim(),
    };
}
function normalizeSituationImage(image, setIndex, imageIndex) {
    if (!image.prompt || image.prompt.trim().length === 0) {
        throw new Error(`Starting set #${setIndex + 1} image #${imageIndex + 1} must include a prompt`);
    }
    const aspect = image.aspect || "landscape";
    if (!["square", "portrait", "landscape"].includes(aspect)) {
        throw new Error(`Starting set #${setIndex + 1} image #${imageIndex + 1} has invalid aspect "${aspect}"`);
    }
    // If no URL is provided, inject a placeholder marker so the client
    // can still render the slot (the app treats cognito sample URLs as placeholders).
    const trimmedUrl = image.imageUrl?.trim() ||
        `https://www.cognitrophy/sampleimage.png?set=${setIndex + 1}&idx=${imageIndex + 1}`;
    return {
        ...(trimmedUrl ? { imageUrl: trimmedUrl } : {}),
        keyword: image.keyword?.trim(),
        situation: image.situation?.trim(),
        aspect: aspect,
        prompt: image.prompt.trim(),
    };
}
function normalizeAdvancedImageSettings(settings) {
    if (!settings) {
        return { ...DEFAULT_ADVANCED_IMAGE_SETTINGS };
    }
    return {
        enabled: settings.enabled ?? true,
        model: settings.model || DEFAULT_ADVANCED_IMAGE_SETTINGS.model,
        sampler: settings.sampler || DEFAULT_ADVANCED_IMAGE_SETTINGS.sampler,
        cfgScale: settings.cfgScale ?? DEFAULT_ADVANCED_IMAGE_SETTINGS.cfgScale,
        steps: settings.steps ?? DEFAULT_ADVANCED_IMAGE_SETTINGS.steps,
        alwaysAddedPrompt: settings.alwaysAddedPrompt?.trim(),
        negativePrompt: settings.negativePrompt?.trim(),
        profilePrompt: settings.profilePrompt?.trim() || "",
        profileAspect: settings.profileAspect || DEFAULT_ADVANCED_IMAGE_SETTINGS.profileAspect,
        // 상황 프롬프트는 startingSets.situationImages에서 자동 생성한다.
        situationPrompts: [],
    };
}
function syncSituationPrompts(character) {
    const prompts = character.startingSets.flatMap((set, setIndex) => (set.situationImages || []).map((image) => ({
        // Image URL is optional; use client-side placeholder when missing
        ...(image.imageUrl
            ? { imageUrl: image.imageUrl }
            : { imageUrl: `sample://placeholder/${setIndex + 1}` }),
        prompt: image.prompt || "",
        aspect: image.aspect || "landscape",
        targets: [setIndex],
        ...(image.keyword ? { keyword: image.keyword } : {}),
        ...(image.situation ? { situation: image.situation } : {}),
    })));
    const filteredPrompts = prompts.filter((prompt) => prompt.prompt.trim().length);
    return {
        ...character,
        advancedImageSettings: {
            ...DEFAULT_ADVANCED_IMAGE_SETTINGS,
            ...character.advancedImageSettings,
            situationPrompts: filteredPrompts,
        },
    };
}
/**
 * Generate default situation images for a starting set
 */
export function generateDefaultSituationImages(setIndex) {
    return [
        {
            keyword: "일상",
            situation: "캐릭터의 일상적인 모습",
            aspect: "landscape",
            prompt: "cozy daily life portrait, natural light, soft focus, gentle smile, warm color palette",
        },
        {
            keyword: "감정",
            situation: "캐릭터의 감정 표현",
            aspect: "portrait",
            prompt: "dramatic close-up, expressive eyes, cinematic lighting, emotional focus, high detail illustration",
        },
        {
            keyword: "특징",
            situation: "캐릭터의 특징적인 순간",
            aspect: "landscape",
            prompt: "dynamic action pose, signature outfit, vibrant accents, motion blur, anime-inspired style",
        },
    ];
}
/**
 * Generate default keyword notes
 */
export function generateDefaultKeywordNotes() {
    return [
        {
            name: "캐릭터 기본 정보",
            prompt: "이 캐릭터의 기본 설정과 배경 정보입니다.",
            keywords: ["캐릭터", "기본"],
            targets: [0],
        },
    ];
}
