import { fetch } from "undici";
import { getApiBaseUrl, getMcpApiKey } from "./config.js";
import { USER_AGENT } from "./version.js";
export class McpApiError extends Error {
    status;
    requestId;
    details;
    constructor(status, message, requestId, details) {
        super(message);
        this.status = status;
        this.requestId = requestId;
        this.details = details;
        this.name = "McpApiError";
    }
}
export async function callMcpApi(path, options = {}) {
    const baseUrl = getApiBaseUrl();
    const normalizedPath = path.startsWith("/") ? path.slice(1) : path;
    const url = new URL(normalizedPath, baseUrl);
    if (options.query) {
        for (const [key, value] of Object.entries(options.query)) {
            if (value === undefined || value === null)
                continue;
            url.searchParams.set(key, String(value));
        }
    }
    const headers = {
        Authorization: `Bearer ${getMcpApiKey()}`,
        "User-Agent": USER_AGENT,
        "X-Client-Version": USER_AGENT,
        Accept: "application/json",
    };
    let body;
    if (options.body !== undefined) {
        headers["Content-Type"] = "application/json";
        body = JSON.stringify(options.body);
    }
    const method = options.method ?? (body ? "POST" : "GET");
    let response;
    try {
        response = await fetch(url, {
            method,
            headers,
            body,
        });
    }
    catch (error) {
        const reason = error instanceof Error ? error.message : String(error);
        throw new McpApiError(-1, `Failed to reach MCP API at ${baseUrl}: ${reason}`);
    }
    const requestId = response.headers.get("x-request-id");
    const rawText = await response.text();
    let data = null;
    if (rawText) {
        try {
            data = JSON.parse(rawText);
        }
        catch {
            // If the response isn't JSON we keep the raw string for error details.
            data = null;
        }
    }
    if (!response.ok) {
        const message = resolveErrorMessage(response.status, data, rawText);
        throw new McpApiError(response.status, message, requestId, data ?? rawText);
    }
    // When a successful response omits the body we return undefined,
    // otherwise the parsed JSON payload.
    return (data ?? undefined);
}
function resolveErrorMessage(status, data, fallbackText) {
    if (typeof data?.error === "string") {
        return data.error;
    }
    if (typeof data?.message === "string") {
        return data.message;
    }
    if (fallbackText) {
        return fallbackText;
    }
    return `Request failed with status ${status}`;
}
